using System;
using System.Collections;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

namespace gov.va.med.vbecs.DAL.HL7AL
{
	/// <summary>
	/// Base client class for connecting to the VistA HL7 application. 
	/// Applications must derive their own specific version of this class.
	/// In order to do this, certain abstract methods must be
	/// implemented with respect to the intended use.
	/// 
	/// Developers are highly encouraged to refer this base class in their code, rather than to 
	/// duplicate implementation. This will allow for easy change in functionality if required.
	/// </summary>
	public abstract class HL7Broker : IDisposable
	{
		private HL7ClientConnection _hl7Connection;
		private bool _isLoggedOn = false;
		private HL7Interface _cachedInterfaceConnectionInfo = null;

//		/// <summary>
//		/// Connection state change event occuring when VBECS HL7 either logged on or off,
//		/// or coldly disconnected. 
//		/// </summary>
//		public event HL7ConnectionStateChangedDelegate ConnectionStateChanged;
	
		/// <summary>
		/// Connects to HL7 Interface listener specified with <see cref="InterfaceConnectionInfo"/> parameter.
		/// </summary>
		/// <param name="interfaceConnectionInfo">VistA server to connect to.</param>
		private void Connect( HL7Interface interfaceConnectionInfo )
		{
			if( interfaceConnectionInfo == null )
				throw( new ArgumentNullException( "interfaceConnectionInfo" ) );

			try
			{
				_hl7Connection = new HL7ClientConnection( interfaceConnectionInfo );
//				_hl7Connection.ConnectionStateChanged += new HL7ConnectionStateChangedDelegate( ConnectionStateChangedHandler );
				_hl7Connection.Connect();
			}
			catch
			{
				Close();
				throw;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8321"> 
		///		<ExpectedInput>This method requires a valid VistA system to connect to and is not testable.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8322"> 
		///		<ExpectedInput>Null HL7Interface input parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Logon to interface specified in <see cref="HL7Interface"/>
		/// </summary>
		/// <param name="interfaceConnectionInfo">Interface configuration information</param>
		/// <returns>bool indicating connection status</returns>
		public bool Logon( HL7Interface interfaceConnectionInfo )
		{
			if( interfaceConnectionInfo == null )
				throw( new ArgumentNullException( "interfaceConnectionInfo" ) );

			return ConnectAndLogon( interfaceConnectionInfo );
		}

		private bool ConnectAndLogon( HL7Interface interfaceConnectionInfo )
		{
			lock( this )
			{
				if( IsLoggedOn )
					throw( new InvalidOperationException( "Already Connected" ) );

				_cachedInterfaceConnectionInfo = interfaceConnectionInfo; // refreshing cached server info because it can be expected by calling code.

				Connect( interfaceConnectionInfo );
				_isLoggedOn = true;

//				OnConnectionStateChanged( new HL7ConnectionStateChangedEventArgs( true ) );

				return _isLoggedOn;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8323"> 
		///		<ExpectedInput>This method requires a valid order message from CPRS and is not testable.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8324"> 
		///		<ExpectedInput>Null HL7ProtocolMessage input parameter.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sends the HL7 message
		/// </summary>
		/// <param name="message">Outbound HL7 message</param>
		/// <returns>Acknowledgement HL7 message</returns>
		public HL7ProtocolMessage ExecuteHL7Send( HL7ProtocolMessage message )
		{
			if( message == null ) 
				throw( new ArgumentNullException( "message" ) );

			return (HL7ProtocolMessage)_hl7Connection.SendReceiveMessage( message );

		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8325"> 
		///		<ExpectedInput>This method requires a valid VistA system with a connection and is not testable.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8326"> 
		///		<ExpectedInput>There are no failure conditions for this method.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Disconnects from HL7 Server. 
		/// </summary>
		public void Close()
		{
			try
			{
				if( _hl7Connection != null )
					_hl7Connection.Close();
			}
			finally
			{
				_hl7Connection = null;				
			}
		}	

		/// <summary>
		/// Shortcut wrapper closing the connection and returning false.
		/// </summary>
		/// <returns>Always returns false.</returns>
		private bool CloseAndReturnFalse()
		{
			Close();

			return false;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8327"> 
		///		<ExpectedInput>This method requires a valid VistA system with a connection and is not testable.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8328"> 
		///		<ExpectedInput>There are no failure conditions for this method.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The property indicates if the <see cref="HL7Broker"/> is connected to the 
		/// remote server. 
		/// </summary>
		public bool IsLoggedOn
		{
			get
			{
				return _hl7Connection != null;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8329"> 
		///		<ExpectedInput>This method requires a valid VistA system with a connection and is not testable.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8330"> 
		///		<ExpectedInput>Broker is not currently logged onto remote server.</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Connection information for active (connected to) remote M server.
		/// </summary>
		public HL7Interface InterfaceConnectionInfo
		{
			get
			{
				if( !IsLoggedOn )
					return null;
				else
					return _hl7Connection.InterfaceConnectionInfo;
			}
		}

		/// <summary>
		/// Standard finalization method.
		/// </summary>
		~HL7Broker()
		{
			Dispose( false );
		}

		/// <summary>
		/// Standard method needed to implement <see cref="IDisposable"/> interface. 
		/// Does some clean-up needed to terminate underlying connection nicely. 
		/// </summary>
		public void Dispose()
		{
			GC.SuppressFinalize( this );
			Dispose( true );
		}

		/// <summary>
		/// Standard implementation of the dispose method. 
		/// </summary>
		/// <param name="disposing">
		///		Flag indicating if the disposition was invoked explicitly under normal 
		///		conditions (true) or forced upon object disposal (false).
		///	</param>
		private void Dispose( bool disposing )
		{
			lock( this )
			{
				try
				{
					if( !disposing )
						return;

					if( _isLoggedOn )
					{
						_isLoggedOn = false;
						Close();
//						OnConnectionStateChanged( new HL7ConnectionStateChangedEventArgs( false ) );
					}

					if( _hl7Connection != null )
					{
//						_hl7Connection.ConnectionStateChanged -= new HL7ConnectionStateChangedDelegate( ConnectionStateChangedHandler );
						_hl7Connection.Close();
					}
				}
				finally
				{
					_isLoggedOn = false;
					_hl7Connection = null;				
				}
			}		
		}

//		/// <summary>
//		/// Handles underlying <see cref="HL7ClientConnection"/> connection state change 
//		/// event and notifies VistALink-consuming classes of connection state change.
//		/// </summary>
//		/// <param name="sender">Event sender object</param>
//		/// <param name="e">Connection state change event arguments.</param>
//		private void ConnectionStateChangedHandler( object sender, HL7ConnectionStateChangedEventArgs e )
//		{
//			if( !e.IsAvailable )
//				Close();			
//		}

//		/// <summary>
//		/// Notifies VistALink-consuming classes of VistALink availability change.
//		/// </summary>
//		/// <param name="e">Connection state change event arguments.</param>
//		protected virtual void OnConnectionStateChanged( HL7ConnectionStateChangedEventArgs e )
//		{
//			if( ConnectionStateChanged != null )
//				ConnectionStateChanged( this, e );
//		}
	}
}
